<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexao com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexao com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexao com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script esta sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'sla_prioridades.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getSla_prioridade($pdo, $id);
            } else {
                getAllSla_prioridades($pdo);
            }
            break;
        case 'POST':
            createSla_prioridade($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateSla_prioridade($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do sla_prioridade nao especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteSla_prioridade($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do sla_prioridade nao especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Metodo nao permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint nao encontrado']);
}

// Funcoes CRUD

function getAllSla_prioridades($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM sla_prioridade");
        $sla_prioridades = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($sla_prioridades);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar sla_prioridades: ' . $e->getMessage()]);
    }
}

function getSla_prioridade($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM sla_prioridade WHERE id = ?");
        $stmt->execute([$id]);
        $sla_prioridade = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($sla_prioridade) {
            echo json_encode($sla_prioridade);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Sla_prioridade nao encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar sla_prioridade: ' . $e->getMessage()]);
    }
}

function createSla_prioridade($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 30) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 30 caracteres']);
        return;
    }

    if (!isset($data['tempo'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "tempo" faltando']);
        return;
    }
    if (strlen($data['tempo']) > 30) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "tempo" deve ter no maximo 30 caracteres']);
        return;
    }


    try {
        $sql = "INSERT INTO sla_prioridade (`descricao`, `tempo`) VALUES (:descricao, :tempo)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':descricao' => $data['descricao'], ':tempo' => $data['tempo']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Sla_prioridade criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar sla_prioridade: ' . $e->getMessage()]);
    }
}

function updateSla_prioridade($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 30) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 30 caracteres']);
        return;
    }

    if (!isset($data['tempo'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "tempo" faltando']);
        return;
    }
    if (strlen($data['tempo']) > 30) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "tempo" deve ter no maximo 30 caracteres']);
        return;
    }


    try {
        // Verifica se o sla_prioridade existe
        $check = $pdo->prepare("SELECT id FROM sla_prioridade WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Sla_prioridade nao encontrado']);
            return;
        }
        
        $sql = "UPDATE sla_prioridade SET `descricao` = :descricao, `tempo` = :tempo WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':descricao' => $data['descricao'], ':tempo' => $data['tempo'], ':id' => $id
        ]);
        
        echo json_encode(['message' => 'Sla_prioridade atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar sla_prioridade: ' . $e->getMessage()]);
    }
}

function deleteSla_prioridade($pdo, $id) {
    try {
        // Verifica se o sla_prioridade existe
        $check = $pdo->prepare("SELECT id FROM sla_prioridade WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Sla_prioridade nao encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM sla_prioridade WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Sla_prioridade deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar sla_prioridade: ' . $e->getMessage()]);
    }
}
?>