<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexao com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexao com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexao com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script esta sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'servico_tipo.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getServico_tipo($pdo, $id);
            } else {
                getAllServico_tipos($pdo);
            }
            break;
        case 'POST':
            createServico_tipo($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateServico_tipo($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do servico_tipo nao especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteServico_tipo($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do servico_tipo nao especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Metodo nao permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint nao encontrado']);
}

// Funcoes CRUD

function getAllServico_tipos($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM servico_tipo");
        $servico_tipos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($servico_tipos);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar servico_tipos: ' . $e->getMessage()]);
    }
}

function getServico_tipo($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM servico_tipo WHERE id = ?");
        $stmt->execute([$id]);
        $servico_tipo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($servico_tipo) {
            echo json_encode($servico_tipo);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Servico_tipo nao encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar servico_tipo: ' . $e->getMessage()]);
    }
}

function createServico_tipo($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 50 caracteres']);
        return;
    }


    try {
        $sql = "INSERT INTO servico_tipo (`descricao`) VALUES (:descricao)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':descricao' => $data['descricao']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Servico_tipo criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar servico_tipo: ' . $e->getMessage()]);
    }
}

function updateServico_tipo($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 50 caracteres']);
        return;
    }


    try {
        // Verifica se o servico_tipo existe
        $check = $pdo->prepare("SELECT id FROM servico_tipo WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Servico_tipo nao encontrado']);
            return;
        }
        
        $sql = "UPDATE servico_tipo SET `descricao` = :descricao WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':descricao' => $data['descricao'], ':id' => $id
        ]);
        
        echo json_encode(['message' => 'Servico_tipo atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar servico_tipo: ' . $e->getMessage()]);
    }
}

function deleteServico_tipo($pdo, $id) {
    try {
        // Verifica se o servico_tipo existe
        $check = $pdo->prepare("SELECT id FROM servico_tipo WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Servico_tipo nao encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM servico_tipo WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Servico_tipo deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar servico_tipo: ' . $e->getMessage()]);
    }
}
?>