<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'contratos_tipo.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getTipoContrato($pdo, $id);
            } else {
                getAllTiposContrato($pdo);
            }
            break;
        case 'POST':
            createTipoContrato($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateTipoContrato($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do tipo de contrato não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteTipoContrato($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do tipo de contrato não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções CRUD

function getAllTiposContrato($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM contrato_tipo ORDER BY descricao");
        $tipos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($tipos);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar tipos de contrato: ' . $e->getMessage()]);
    }
}

function getTipoContrato($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM contrato_tipo WHERE id = ?");
        $stmt->execute([$id]);
        $tipo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($tipo) {
            echo json_encode($tipo);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Tipo de contrato não encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar tipo de contrato: ' . $e->getMessage()]);
    }
}

function createTipoContrato($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || !isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos ou campo "descricao" faltando']);
        return;
    }
    
    // Validação do tamanho máximo da descrição
    if (strlen($data['descricao']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'A descrição deve ter no máximo 50 caracteres']);
        return;
    }
    
    try {
        $sql = "INSERT INTO contrato_tipo (descricao) VALUES (:descricao)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':descricao' => $data['descricao']]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Tipo de contrato criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar tipo de contrato: ' . $e->getMessage()]);
    }
}

function updateTipoContrato($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || !isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos ou campo "descricao" faltando']);
        return;
    }
    
    // Validação do tamanho máximo da descrição
    if (strlen($data['descricao']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'A descrição deve ter no máximo 50 caracteres']);
        return;
    }
    
    try {
        // Verifica se o tipo de contrato existe
        $check = $pdo->prepare("SELECT id FROM contrato_tipo WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Tipo de contrato não encontrado']);
            return;
        }
        
        $sql = "UPDATE contrato_tipo SET descricao = :descricao WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':descricao' => $data['descricao']
        ]);
        
        echo json_encode(['message' => 'Tipo de contrato atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar tipo de contrato: ' . $e->getMessage()]);
    }
}

function deleteTipoContrato($pdo, $id) {
    try {
        // Verifica se o tipo de contrato existe
        $check = $pdo->prepare("SELECT id FROM contrato_tipo WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Tipo de contrato não encontrado']);
            return;
        }
        
        // Verificar se o tipo está sendo usado em algum contrato antes de deletar
        // (Adicione esta verificação se a tabela contrato_tipo for referenciada em outras tabelas)
        
        $stmt = $pdo->prepare("DELETE FROM contrato_tipo WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Tipo de contrato deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar tipo de contrato: ' . $e->getMessage()]);
    }
}
?>