<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'contrato_maquina_leituras.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getLeitura($pdo, $id);
            } else {
                $fk_contrato_makina = isset($_GET['fk_contrato_makina']) ? (int)$_GET['fk_contrato_makina'] : null;
                getAllLeituras($pdo, $fk_contrato_makina);
            }
            break;
        case 'POST':
            createLeitura($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateLeitura($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID da leitura não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteLeitura($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID da leitura não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções CRUD

function getAllLeituras($pdo, $fk_contrato_makina = null) {
    try {
        $sql = "SELECT * FROM contrato_maquina_leituras";
        $params = [];
        
        if ($fk_contrato_makina) {
            $sql .= " WHERE fk_contrato_makina = ?";
            $params[] = $fk_contrato_makina;
        }
        
        $sql .= " ORDER BY leitura_data DESC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        $leituras = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($leituras);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar leituras: ' . $e->getMessage()]);
    }
}

function getLeitura($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM contrato_maquina_leituras WHERE id = ?");
        $stmt->execute([$id]);
        $leitura = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($leitura) {
            echo json_encode($leitura);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Leitura não encontrada']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar leitura: ' . $e->getMessage()]);
    }
}

function createLeitura($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Validação dos campos obrigatórios
    $requiredFields = ['fk_contrato_makina', 'serie', 'leitura_data'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Campo '$field' é obrigatório"]);
            return;
        }
    }
    
    // Validação dos tamanhos máximos
    if (strlen($data['serie']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "serie" deve ter no máximo 50 caracteres']);
        return;
    }
    
    try {
        $sql = "INSERT INTO contrato_maquina_leituras 
                (fk_contrato_makina, serie, leitura_data, contador_pb, contador_color, contador_digit) 
                VALUES (:fk_contrato_makina, :serie, :leitura_data, :contador_pb, :contador_color, :contador_digit)";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_contrato_makina' => $data['fk_contrato_makina'],
            ':serie' => $data['serie'],
            ':leitura_data' => $data['leitura_data'],
            ':contador_pb' => $data['contador_pb'] ?? null,
            ':contador_color' => $data['contador_color'] ?? null,
            ':contador_digit' => $data['contador_digit'] ?? null
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Leitura criada com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar leitura: ' . $e->getMessage()]);
    }
}

function updateLeitura($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Validação dos campos obrigatórios
    $requiredFields = ['fk_contrato_makina', 'serie', 'leitura_data'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Campo '$field' é obrigatório"]);
            return;
        }
    }
    
    // Validação dos tamanhos máximos
    if (strlen($data['serie']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "serie" deve ter no máximo 50 caracteres']);
        return;
    }
    
    try {
        // Verifica se a leitura existe
        $check = $pdo->prepare("SELECT id FROM contrato_maquina_leituras WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Leitura não encontrada']);
            return;
        }
        
        $sql = "UPDATE contrato_maquina_leituras SET 
                fk_contrato_makina = :fk_contrato_makina,
                serie = :serie,
                leitura_data = :leitura_data,
                contador_pb = :contador_pb,
                contador_color = :contador_color,
                contador_digit = :contador_digit
                WHERE id = :id";
                
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':fk_contrato_makina' => $data['fk_contrato_makina'],
            ':serie' => $data['serie'],
            ':leitura_data' => $data['leitura_data'],
            ':contador_pb' => $data['contador_pb'] ?? null,
            ':contador_color' => $data['contador_color'] ?? null,
            ':contador_digit' => $data['contador_digit'] ?? null
        ]);
        
        echo json_encode(['message' => 'Leitura atualizada com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar leitura: ' . $e->getMessage()]);
    }
}

function deleteLeitura($pdo, $id) {
    try {
        // Verifica se a leitura existe
        $check = $pdo->prepare("SELECT id FROM contrato_maquina_leituras WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Leitura não encontrada']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM contrato_maquina_leituras WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Leitura deletada com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar leitura: ' . $e->getMessage()]);
    }
}
?>