<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;
$chamadoId = isset($_GET['chamado_id']) ? (int)$_GET['chamado_id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'chamado_itens.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getItem($pdo, $id);
            } else if ($chamadoId) {
                getItensByChamado($pdo, $chamadoId);
            } else {
                getAllItens($pdo);
            }
            break;
        case 'POST':
            createItem($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateItem($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do item não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteItem($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do item não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções para manipulação dos itens de chamado

function getAllItens($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM chamado_atendimento");
        $itens = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($itens);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar itens: ' . $e->getMessage()]);
    }
}

function getItensByChamado($pdo, $chamadoId) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM chamado_atendimento WHERE fk_chamado = ?");
        $stmt->execute([$chamadoId]);
        $itens = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($itens);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar itens: ' . $e->getMessage()]);
    }
}

function getItem($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM chamado_atendimento WHERE id = ?");
        $stmt->execute([$id]);
        $item = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($item) {
            echo json_encode($item);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Item não encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar item: ' . $e->getMessage()]);
    }
}

function createItem($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $requiredFields = ['fk_chamado', 'tecnico', 'prioridade', 'ocorrencia', 'status'];
    
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Campo '$field' é obrigatório"]);
            return;
        }
    }
    
    try {
        // Verifica se o chamado existe
        $checkChamado = $pdo->prepare("SELECT id FROM chamado WHERE id = ?");
        $checkChamado->execute([$data['fk_chamado']]);
        
        if (!$checkChamado->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Chamado não encontrado']);
            return;
        }
        
        $sql = "INSERT INTO chamado_atendimento (fk_chamado, tecnico, prioridade, ocorrencia, 
                atendimento, evento, tempo, contador_pb, contador_color, contador_total, 
                scanner_total, anexo, status)
                VALUES (:fk_chamado, :tecnico, :prioridade, :ocorrencia, :atendimento, 
                :evento, :tempo, :contador_pb, :contador_color, :contador_total, 
                :scanner_total, :anexo, :status)";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_chamado' => $data['fk_chamado'],
            ':tecnico' => $data['tecnico'],
            ':prioridade' => $data['prioridade'],
            ':ocorrencia' => $data['ocorrencia'],
            ':atendimento' => $data['atendimento'] ?? null,
            ':evento' => $data['evento'] ?? null,
            ':tempo' => $data['tempo'] ?? null,
            ':contador_pb' => $data['contador_pb'] ?? null,
            ':contador_color' => $data['contador_color'] ?? null,
            ':contador_total' => $data['contador_total'] ?? null,
            ':scanner_total' => $data['scanner_total'] ?? null,
            ':anexo' => $data['anexo'] ?? null,
            ':status' => $data['status']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Item criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar item: ' . $e->getMessage()]);
    }
}

function updateItem($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos']);
        return;
    }
    
    try {
        // Verifica se o item existe
        $check = $pdo->prepare("SELECT id FROM chamado_atendimento WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Item não encontrado']);
            return;
        }
        
        // Monta a query dinamicamente com os campos fornecidos
        $fields = [];
        $params = [':id' => $id];
        
        $allowedFields = ['fk_chamado', 'tecnico', 'prioridade', 'ocorrencia', 'atendimento', 
                        'evento', 'tempo', 'contador_pb', 'contador_color', 'contador_total', 
                        'scanner_total', 'anexo', 'status'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = :$field";
                $params[":$field"] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            http_response_code(400);
            echo json_encode(['error' => 'Nenhum campo válido para atualização']);
            return;
        }
        
        // Se fk_chamado foi alterado, verifica se existe
        if (isset($data['fk_chamado'])) {
            $checkChamado = $pdo->prepare("SELECT id FROM chamado WHERE id = ?");
            $checkChamado->execute([$data['fk_chamado']]);
            
            if (!$checkChamado->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Chamado não encontrado']);
                return;
            }
        }
        
        $sql = "UPDATE chamado_atendimento SET " . implode(', ', $fields) . " WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        echo json_encode(['message' => 'Item atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar item: ' . $e->getMessage()]);
    }
}

function deleteItem($pdo, $id) {
    try {
        // Verifica se o item existe
        $check = $pdo->prepare("SELECT id FROM chamado_atendimento WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Item não encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM chamado_atendimento WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Item deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar item: ' . $e->getMessage()]);
    }
}