<?php
$api_url = 'http://localhost/erp/backend/routers/produtos';
$produtos = json_decode(file_get_contents($api_url), true);

function formatCurrency($value) {
    return 'R$ ' . number_format(floatval($value), 2, ',', '.');
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestão de Produtos</title>
    <link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/v/bs5/jq-3.6.0/dt-1.11.5/datatables.min.css"/>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .btn-action {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
            line-height: 1.5;
        }
        .modal-lg-custom {
            max-width: 800px;
        }
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
        <h1 class="mb-4">Gestão de Produtos</h1>
        
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Lista de Produtos</h5>
                <div>
                    <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#cadastrarModal">
                        <i class="fas fa-plus me-1"></i> Cadastrar
                    </button>
                </div>
            </div>
            <div class="card-body">
                <table id="produtosTable" class="table table-striped table-bordered nowrap" style="width:100%">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Foto</th>
                            <th>Tipo</th>
                            <th>Unidade</th>
                            <th>NCM</th>
                            <th>Produto</th>
                            <th>Fabricante</th>
                            <th>Modelo</th>
                            <th>Custo</th>
                            <th>Status</th>
                            <th>Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($produtos as $produto): ?>
                        <tr>
                            <td><?= $produto['id'] ?></td>
                            <td>
                                <?php if ($produto['foto']): ?>
                                    <img src="<?= $produto['foto'] ?>" alt="Foto" style="max-width: 50px;">
                                <?php else: ?>
                                    <span class="text-muted">Sem foto</span>
                                <?php endif; ?>
                            </td>
                            <td><?= $produto['tipo'] ?></td>
                            <td><?= $produto['un'] ?></td>
                            <td><?= $produto['ncm'] ?></td>
                            <td><?= $produto['produto'] ?></td>
                            <td><?= $produto['fabricante'] ?></td>
                            <td><?= $produto['modelo'] ?></td>
                            <td><?= formatCurrency($produto['custo']) ?></td>
                            <td>
                                <span class="badge bg-<?= $produto['status'] == 'Ativo' ? 'success' : 'danger' ?>">
                                    <?= $produto['status'] ?>
                                </span>
                            </td>
                            <td>
                                <button class="btn btn-info btn-action btn-view" data-id="<?= $produto['id'] ?>">
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button class="btn btn-warning btn-action btn-edit" data-id="<?= $produto['id'] ?>">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-danger btn-action btn-delete" data-id="<?= $produto['id'] ?>">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Modal Visualizar -->
    <div class="modal fade" id="viewModal" tabindex="-1" aria-labelledby="viewModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-lg-custom">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="viewModalLabel">Detalhes do Produto</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="viewModalBody">
                    <!-- Conteúdo será preenchido via AJAX -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal Editar -->
    <div class="modal fade" id="editModal" tabindex="-1" aria-labelledby="editModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-lg-custom">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="editModalLabel">Editar Produto</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="editForm" method="POST">
                    <div class="modal-body" id="editModalBody">
                        <!-- Conteúdo será preenchido via AJAX -->
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Salvar Alterações</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Excluir -->
    <div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="deleteModalLabel">Confirmar Exclusão</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p>Você tem certeza que deseja excluir este produto? Esta ação não pode ser desfeita.</p>
                    <input type="hidden" id="deleteId">
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="button" class="btn btn-danger" id="confirmDelete">Excluir</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal Cadastrar -->
    <div class="modal fade" id="cadastrarModal" tabindex="-1" aria-labelledby="cadastrarModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-lg-custom">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="cadastrarModalLabel">Cadastrar Novo Produto</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="cadastrarForm" method="POST" action="<?= $api_url ?>" enctype="multipart/form-data">
                    <div class="modal-body">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label for="produto" class="form-label">Nome do Produto</label>
                                <input type="text" class="form-control" id="produto" name="produto" required>
                            </div>
                            <div class="col-md-6">
                                <label for="tipo" class="form-label">Tipo</label>
                                <input type="text" class="form-control" id="tipo" name="tipo" required>
                            </div>
                            <div class="col-md-4">
                                <label for="un" class="form-label">Unidade</label>
                                <input type="text" class="form-control" id="un" name="un" required>
                            </div>
                            <div class="col-md-4">
                                <label for="ncm" class="form-label">NCM</label>
                                <input type="text" class="form-control" id="ncm" name="ncm" required>
                            </div>
                            <div class="col-md-4">
                                <label for="fabricante" class="form-label">Fabricante</label>
                                <input type="text" class="form-control" id="fabricante" name="fabricante" required>
                            </div>
                            <div class="col-md-6">
                                <label for="modelo" class="form-label">Modelo</label>
                                <input type="text" class="form-control" id="modelo" name="modelo" required>
                            </div>
                            <div class="col-md-6">
                                <label for="status" class="form-label">Status</label>
                                <select class="form-select" id="status" name="status" required>
                                    <option value="Ativo">Ativo</option>
                                    <option value="Inativo">Inativo</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label for="custo" class="form-label">Custo</label>
                                <input type="number" step="0.01" class="form-control" id="custo" name="custo" required>
                            </div>
                            <div class="col-md-6">
                                <label for="venda" class="form-label">Preço de Venda</label>
                                <input type="number" step="0.01" class="form-control" id="venda" name="venda">
                            </div>
                            <div class="col-12">
                                <label for="descricao" class="form-label">Descrição</label>
                                <textarea class="form-control" id="descricao" name="descricao" rows="3"></textarea>
                            </div>
                            <div class="col-md-6">
                                <label for="armazem" class="form-label">Armazém</label>
                                <input type="text" class="form-control" id="armazem" name="armazem">
                            </div>
                            <div class="col-md-6">
                                <label for="local" class="form-label">Local</label>
                                <input type="text" class="form-control" id="local" name="local">
                            </div>
                            <div class="col-md-6">
                                <label for="localizacao" class="form-label">Localização</label>
                                <input type="text" class="form-control" id="localizacao" name="localizacao">
                            </div>
                            <div class="col-md-6">
                                <label for="for_pad" class="form-label">Fornecedor Padrão</label>
                                <input type="text" class="form-control" id="for_pad" name="for_pad">
                            </div>
                            <div class="col-md-6">
                                <label for="qtd_min" class="form-label">Quantidade Mínima</label>
                                <input type="number" class="form-control" id="qtd_min" name="qtd_min">
                            </div>
                            <div class="col-md-6">
                                <label for="qtd_max" class="form-label">Quantidade Máxima</label>
                                <input type="number" class="form-control" id="qtd_max" name="qtd_max">
                            </div>
                            <div class="col-12">
                                <label for="foto" class="form-label">Foto do Produto</label>
                                <input type="file" class="form-control" id="foto" name="foto">
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Cadastrar</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- jQuery, Bootstrap e DataTables JS -->
    <script type="text/javascript" src="https://cdn.datatables.net/v/bs5/jq-3.6.0/dt-1.11.5/datatables.min.js"></script>
    <!-- Excel export -->
    <script src="https://cdn.datatables.net/buttons/2.2.2/js/dataTables.buttons.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.1.3/jszip.min.js"></script>
    <script src="https://cdn.datatables.net/buttons/2.2.2/js/buttons.html5.min.js"></script>

    <script>
        $(document).ready(function() {
    // Inicialização do DataTable com botão de exportação
    var table = $('#produtosTable').DataTable({
        responsive: true,
        dom: 'Bfrtip',
        buttons: [
            {
                extend: 'excel',
                text: '<i class="fas fa-file-excel me-1"></i> Exportar Excel',
                className: 'btn btn-success btn-sm',
                title: 'Produtos',
                exportOptions: {
                    columns: ':not(:last-child)' // Exclui a coluna de ações
                }
            }
        ],
        language: {
            url: 'https://cdn.datatables.net/plug-ins/1.11.5/i18n/pt-BR.json'
        }
    });

    // Move o botão de exportação para o local correto
    table.buttons().container().appendTo('#exportExcel');

    // Visualizar produto
    $(document).on('click', '.btn-view', function() {
        var id = $(this).data('id');
        $.ajax({
            url: '<?= $api_url ?>?id=' + id,
            type: 'GET',
            dataType: 'json',
            success: function(produto) {
                var html = `
                    <div class="row">
                        <div class="col-md-4">
                            <h6>Informações Básicas</h6>
                            <p><strong>ID:</strong> ${produto.id}</p>
                            <p><strong>Produto:</strong> ${produto.produto}</p>
                            <p><strong>Tipo:</strong> ${produto.tipo}</p>
                            <p><strong>Status:</strong> <span class="badge bg-${produto.status == 'Ativo' ? 'success' : 'danger'}">${produto.status}</span></p>
                        </div>
                        <div class="col-md-4">
                            <h6>Detalhes</h6>
                            <p><strong>Fabricante:</strong> ${produto.fabricante}</p>
                            <p><strong>Modelo:</strong> ${produto.modelo}</p>
                            <p><strong>NCM:</strong> ${produto.ncm}</p>
                            <p><strong>Unidade:</strong> ${produto.un}</p>
                        </div>
                        <div class="col-md-4">
                            <h6>Valores</h6>
                            <p><strong>Custo:</strong> ${formatCurrency(produto.custo)}</p>
                            <p><strong>Preço de Venda:</strong> ${produto.venda ? formatCurrency(produto.venda) : 'Não definido'}</p>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-12">
                            <h6>Descrição</h6>
                            <p>${produto.descricao || 'Nenhuma descrição disponível.'}</p>
                        </div>
                    </div>
                `;
                $('#viewModalBody').html(html);
                $('#viewModal').modal('show');
            },
            error: function(xhr, status, error) {
                alert('Erro ao carregar produto: ' + error);
            }
        });
    });

    // Editar produto - Abrir modal
    $(document).on('click', '.btn-edit', function() {
        var id = $(this).data('id');
        $.ajax({
            url: '<?= $api_url ?>?id=' + id,
            type: 'GET',
            dataType: 'json',
            success: function(produto) {
                var html = `
                    <input type="hidden" name="id" value="${produto.id}">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label for="edit_produto" class="form-label">Nome do Produto</label>
                            <input type="text" class="form-control" id="edit_produto" name="produto" value="${produto.produto || ''}" required>
                        </div>
                        <div class="col-md-6">
                            <label for="edit_tipo" class="form-label">Tipo</label>
                            <input type="text" class="form-control" id="edit_tipo" name="tipo" value="${produto.tipo || ''}" required>
                        </div>
                        <div class="col-md-4">
                            <label for="edit_un" class="form-label">Unidade</label>
                            <input type="text" class="form-control" id="edit_un" name="un" value="${produto.un || ''}" required>
                        </div>
                        <div class="col-md-4">
                            <label for="edit_ncm" class="form-label">NCM</label>
                            <input type="text" class="form-control" id="edit_ncm" name="ncm" value="${produto.ncm || ''}" required>
                        </div>
                        <div class="col-md-4">
                            <label for="edit_fabricante" class="form-label">Fabricante</label>
                            <input type="text" class="form-control" id="edit_fabricante" name="fabricante" value="${produto.fabricante || ''}" required>
                        </div>
                        <div class="col-md-6">
                            <label for="edit_modelo" class="form-label">Modelo</label>
                            <input type="text" class="form-control" id="edit_modelo" name="modelo" value="${produto.modelo || ''}" required>
                        </div>
                        <div class="col-md-6">
                            <label for="edit_status" class="form-label">Status</label>
                            <select class="form-select" id="edit_status" name="status" required>
                                <option value="Ativo" ${produto.status == 'Ativo' ? 'selected' : ''}>Ativo</option>
                                <option value="Inativo" ${produto.status == 'Inativo' ? 'selected' : ''}>Inativo</option>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label for="edit_custo" class="form-label">Custo</label>
                            <input type="number" step="0.01" class="form-control" id="edit_custo" name="custo" value="${produto.custo || ''}" required>
                        </div>
                        <div class="col-md-6">
                            <label for="edit_venda" class="form-label">Preço de Venda</label>
                            <input type="number" step="0.01" class="form-control" id="edit_venda" name="venda" value="${produto.venda || ''}">
                        </div>
                        <div class="col-12">
                            <label for="edit_descricao" class="form-label">Descrição</label>
                            <textarea class="form-control" id="edit_descricao" name="descricao" rows="3">${produto.descricao || ''}</textarea>
                        </div>
                    </div>
                `;
                $('#editModalBody').html(html);
                $('#editModal').modal('show');
            },
            error: function(xhr, status, error) {
                alert('Erro ao carregar produto para edição: ' + error);
            }
        });
    });

    // Submit do formulário de edição
    $('#editForm').submit(function(e) {
        e.preventDefault();
        var id = $('input[name="id"]').val();
        var formData = {
            produto: $('#edit_produto').val(),
            tipo: $('#edit_tipo').val(),
            un: $('#edit_un').val(),
            ncm: $('#edit_ncm').val(),
            fabricante: $('#edit_fabricante').val(),
            modelo: $('#edit_modelo').val(),
            status: $('#edit_status').val(),
            custo: $('#edit_custo').val(),
            venda: $('#edit_venda').val(),
            descricao: $('#edit_descricao').val()
        };
        
        $.ajax({
            url: '<?= $api_url ?>?id=' + id,
            type: 'PUT',
            contentType: 'application/json',
            data: JSON.stringify(formData),
            success: function(response) {
                alert('Produto atualizado com sucesso!');
                $('#editModal').modal('hide');
                location.reload();
            },
            error: function(xhr, status, error) {
                alert('Erro ao atualizar o produto: ' + (xhr.responseJSON?.error || error));
            }
        });
    });

    // Excluir produto
    $(document).on('click', '.btn-delete', function() {
        var id = $(this).data('id');
        $('#deleteId').val(id);
        $('#deleteModal').modal('show');
    });

    // Confirmar exclusão
    $('#confirmDelete').click(function() {
        var id = $('#deleteId').val();
        
        $.ajax({
            url: '<?= $api_url ?>?id=' + id,
            type: 'DELETE',
            success: function(response) {
                alert('Produto excluído com sucesso!');
                $('#deleteModal').modal('hide');
                location.reload();
            },
            error: function(xhr, status, error) {
                alert('Erro ao excluir o produto: ' + (xhr.responseJSON?.error || error));
            }
        });
    });

    // Submit do formulário de cadastro
    $('#cadastrarForm').submit(function(e) {
        e.preventDefault();
        var formData = {
            produto: $('#produto').val(),
            tipo: $('#tipo').val(),
            un: $('#un').val(),
            ncm: $('#ncm').val(),
            fabricante: $('#fabricante').val(),
            modelo: $('#modelo').val(),
            status: $('#status').val(),
            custo: $('#custo').val(),
            venda: $('#venda').val(),
            descricao: $('#descricao').val(),
            armazem: $('#armazem').val(),
            local: $('#local').val(),
            localizacao: $('#localizacao').val(),
            qtd_min: $('#qtd_min').val(),
            qtd_max: $('#qtd_max').val(),
            for_pad: $('#for_pad').val()
        };
        
        $.ajax({
            url: '<?= $api_url ?>',
            type: 'POST',
            contentType: 'application/json',
            data: JSON.stringify(formData),
            success: function(response) {
                alert('Produto cadastrado com sucesso!');
                $('#cadastrarModal').modal('hide');
                location.reload();
            },
            error: function(xhr, status, error) {
                alert('Erro ao cadastrar o produto: ' + (xhr.responseJSON?.error || error));
            }
        });
    });

    // Função para formatar moeda
    function formatCurrency(value) {
        if (!value) return 'R$ 0,00';
        return 'R$ ' + parseFloat(value).toFixed(2).replace('.', ',').replace(/(\d)(?=(\d{3})+\,)/g, "$1.");
    }
});
    </script>
</body>
</html>