<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'eqp_fabricantes.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getFabricante($pdo, $id);
            } else {
                getAllFabricantes($pdo);
            }
            break;
        case 'POST':
            createFabricante($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateFabricante($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do fabricante não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteFabricante($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do fabricante não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções CRUD

function getAllFabricantes($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM eq_fabricantes ORDER BY nome");
        $fabricantes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($fabricantes);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar fabricantes: ' . $e->getMessage()]);
    }
}

function getFabricante($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM eq_fabricantes WHERE id = ?");
        $stmt->execute([$id]);
        $fabricante = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($fabricante) {
            echo json_encode($fabricante);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Fabricante não encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar fabricante: ' . $e->getMessage()]);
    }
}

function createFabricante($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || !isset($data['nome'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos ou campo "nome" faltando']);
        return;
    }
    
    // Validação do tamanho máximo do nome
    if (strlen($data['nome']) > 200) {
        http_response_code(400);
        echo json_encode(['error' => 'O nome do fabricante deve ter no máximo 200 caracteres']);
        return;
    }
    
    try {
        $sql = "INSERT INTO eq_fabricantes (nome) VALUES (:nome)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':nome' => $data['nome']]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Fabricante criado com sucesso']);
    } catch (PDOException $e) {
        // Verifica se é erro de duplicação (nome único)
        if ($e->getCode() == 23000) {
            http_response_code(400);
            echo json_encode(['error' => 'Já existe um fabricante com este nome']);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao criar fabricante: ' . $e->getMessage()]);
        }
    }
}

function updateFabricante($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || !isset($data['nome'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos ou campo "nome" faltando']);
        return;
    }
    
    // Validação do tamanho máximo do nome
    if (strlen($data['nome']) > 200) {
        http_response_code(400);
        echo json_encode(['error' => 'O nome do fabricante deve ter no máximo 200 caracteres']);
        return;
    }
    
    try {
        // Verifica se o fabricante existe
        $check = $pdo->prepare("SELECT id FROM eq_fabricantes WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Fabricante não encontrado']);
            return;
        }
        
        $sql = "UPDATE eq_fabricantes SET nome = :nome WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':nome' => $data['nome']
        ]);
        
        echo json_encode(['message' => 'Fabricante atualizado com sucesso']);
    } catch (PDOException $e) {
        // Verifica se é erro de duplicação (nome único)
        if ($e->getCode() == 23000) {
            http_response_code(400);
            echo json_encode(['error' => 'Já existe um fabricante com este nome']);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao atualizar fabricante: ' . $e->getMessage()]);
        }
    }
}

function deleteFabricante($pdo, $id) {
    try {
        // Verifica se o fabricante existe
        $check = $pdo->prepare("SELECT id FROM eq_fabricantes WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Fabricante não encontrado']);
            return;
        }
        
        // Verifica se o fabricante está sendo usado em algum modelo
        $checkUsage = $pdo->prepare("SELECT id FROM eq_modelos WHERE id_fabricante = ? LIMIT 1");
        $checkUsage->execute([$id]);
        
        if ($checkUsage->fetch()) {
            http_response_code(400);
            echo json_encode(['error' => 'Este fabricante está associado a um ou mais modelos e não pode ser excluído']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM eq_fabricantes WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Fabricante deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar fabricante: ' . $e->getMessage()]);
    }
}
?>